// App.jsx — versión limpia y robusta (solo WebView + Login simple + Notif. Push) - en emulador expo puede invadir el notif bar y los iconos pero no al compilar...
import React, {
    useEffect,
    useRef,
    useState
} from "react";
import {
    ActivityIndicator,
    Image,
    SafeAreaView,
    StyleSheet,
    Text,
    TextInput,
    TouchableOpacity,
    View,
    KeyboardAvoidingView,
    Platform,
    Animated,
    Easing
} from "react-native";
import AsyncStorage from "@react-native-async-storage/async-storage";
import { WebView } from "react-native-webview"; //El modulo web embebido
//import CookieManager from "@react-native-cookies/cookies"; // Borrar sesión cookies del webview, aunque mejor por php
import { registerRootComponent } from "expo"; //Permite solo sea un archivo en raiz, registra el componente raíz de la app


// =====================
// CONFIGURA AQUÍ, la app en si es un navegador que guarda el login en la app en asyncstorage y tiene notif push, pero todo es web
// =====================
const API_LOGIN_URL = "https://yonduur.com/app-login.php"; //Endpoint login, regresa id user para cookie
const API_REGISTER_URL = "https://yonduur.com/app-register.php"; //Endpoint nuevos usuarios, regresa id para cookie
const LOGOUT_URL = "https://yonduur.com/app-logout.php"; // URL que dispara el logout aunque este en blanco
const WEBAPP_URL = "https://yonduur.com/listings/for-rent"; //El home de tu app en la pag web, alla ya pasa toda la magia
const COOKIE_DAYS = 30; //Cada cuantos días se cierra sesion

export default function App() {
    // Estados
    const [mode, setMode] = useState("login"); // "login" | "register"
    const [email, setEmail] = useState("");
    const [name, setName] = useState("");
    const [password, setPassword] = useState("");
    const [showPwd, setShowPwd] = useState(false); //toggle de visibilidad del password
    const [loading, setLoading] = useState(false);
    const [userId, setUserId] = useState(null);

    const [didInitReload, setDidInitReload] = useState(false);
    const webRef = useRef(null);
    const [errorMsg, setErrorMsg] = useState("");
    const [hasNetError, setHasNetError] = useState(false);
    const [navKey, setNavKey] = useState(0); // para forzar remount del WebView
    const retry = () => {
        setHasNetError(false);
        setWvLoading(true);
        fade.setValue(0);
        webRef.current?.reload();
    };
    const goHome = () => {
        setHasNetError(false);
        setWvLoading(true);
        fade.setValue(0);
        setNavKey(k => k + 1); // remonta el WebView en WEBAPP_URL
    };
    
    const [wvLoading, setWvLoading] = useState(true);//no se note cambio entre cargas pagina
    const fade = useRef(new Animated.Value(0)).current;
    
    // Cerrar sesión (borra cookies del WebView y limpia storage)
    const clientSideLogout = async () => {
        // Borrar todas las cookies del contenedor WebView, elimina toda la sesion
        //await CookieManager.clearAll(); //ni sirve ni se usa esta libreria
        console.log("Todas las cookies eliminadas pero desde php para facil");
                        
        try { await AsyncStorage.removeItem("@userId"); } catch (_) {}
        // Resetea UI a login y limpia inputs
        setDidInitReload(false);
        setUserId(null);         // regresa a la pantalla de login
        setMode("login");
        setEmail("");
        setPassword("");
        setName("");
        setShowPwd(false);
        setErrorMsg("Logged out");
    };
    // Función para arrancar el fade-in al terminar la carga
    const runFadeIn = () => {
        Animated.timing(fade, {
            toValue: 1,
            duration: 220, // transición corta y suave
            easing: Easing.out(Easing.quad),
            useNativeDriver: true
        }).start(() => setWvLoading(false)); // al final quitamos el overlay
    };
            
    // Autocierra el banner de error después de 4s
    useEffect(() => {
        if (!errorMsg) return;
        const t = setTimeout(() => setErrorMsg(""), 4000);
        return () => clearTimeout(t);
    }, [errorMsg]);
    
    // Recupera sesión previa
    useEffect(() => {
        (async () => {
            try {
                const saved = await AsyncStorage.getItem("@userId");
                if (saved) setUserId(saved);
            } catch (e) {
                console.warn("AsyncStorage getItem error:", e);
            }
        })();
    }, []);

    // Endpoint espera JSON; NO reintentar en form-urlencoded para no ocultar errores 401
    async function parseErrorResponse(resp) {
        try {
            // intenta parsear como JSON primero
            const data = await resp.json();
            // si tu endpoint devuelve { error: "...", message: "..." }
            if (data.error) return data.error;
            if (data.message) return data.message;
            // si no hay campos conocidos, muestra el status
            return 'HTTP ' + resp.status;
        } catch {
            // si no era JSON válido, intenta como texto simple
            try {
                const txt = await resp.text();
                return txt || 'HTTP ' + resp.status;
            } catch {
                return 'HTTP ' + resp.status;
            }
        }
    }
    async function postWithFallback(url, payload) {
        const resp = await fetch(url, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(payload),
        });
        return resp; // devolvemos la primera respuesta tal cual
    }        
    const isEmail = (s = "") => /\S+@\S+\.\S+/.test(s);
    const doAuth = async () => {
        try {
            if (!isEmail(email)) {
                 setErrorMsg("Invalid email format");
                 return;
            }
            if (password.length < 3) {
                setErrorMsg("Password must be at least 3 characters");
                return;
            }
            if (mode === "register" && name.trim().length < 2) {
                setErrorMsg("Name is required");
                return;
            }
            setLoading(true);

            const url = mode === "login" ? API_LOGIN_URL : API_REGISTER_URL;
            const payload = {
                email,
                password,
                ...(mode === "register" ? {
                    name
                } : {})
            };

            const resp = await postWithFallback(url, payload);
            if (!resp.ok) {
                const msg = await parseErrorResponse(resp);
                throw new Error(msg || "Server error");
            }

            // Permite que el PHP regrese { userId: 123 } o { id: 123 }
            let json = {};
            try {
                json = await resp.json();
            } catch {
                // Si el backend devuelve texto plano "123", lo tomamos
                const asText = await resp.text();
                json = {
                    userId: asText?.trim()
                };
            }

            const returnedId = String(json.userId ?? json.id ?? "").trim();
            if (!returnedId) throw new Error("No userId in response");

            await AsyncStorage.setItem("@userId", returnedId);
            setErrorMsg(""); // limpia error visual si había
            setUserId(returnedId);
                    

        } catch (e) {
            console.warn("Auth error:", e);
            const msg = (e?.message || "").toString();
            // Por si no hay internet
            if (msg.includes("Network request failed")) {
               setErrorMsg("No internet connection");
            } else {
               setErrorMsg(msg);
            }
        } finally {
            setLoading(false);
        }
    };
                    
    // elimina el reload visible. Ya mandas la cookie en el primer request con headers y además la inyectas con injectedJavaScriptBeforeContentLoaded. No hace falta recargar.
      const handleCookieAck = () => {
          if (!didInitReload) {
              setDidInitReload(true);
              webRef.current?.reload(); // eliminado para evitar parpadeo
          }
      };

      // Pantalla offline
      if (hasNetError) {
        return (
          <SafeAreaView style={styles.safe}>
            <View style={styles.offlineWrap}>
              <Text style={styles.offlineTitle}>Connection broken</Text>
              <Text style={styles.offlineText}>Please check your internet connection.</Text>
              <TouchableOpacity style={styles.btn} onPress={retry}>
                <Text style={styles.btnText}>Retry</Text>
              </TouchableOpacity>
            </View>
          </SafeAreaView>
        );
      }

      // Vista Web (usuario autenticado)
      if (userId) {
        return (
          <View style={{ flex: 1 }}>

            {/* overlay de carga para ocultar el “flash” inicial */}
            {wvLoading && (
              <View style={styles.loadingOverlay}>
                <ActivityIndicator size="large" />
                <Text style={{ marginTop: 8 }}>Loading…</Text>
              </View>
            )}

            {/* envolvemos el WebView en un Animated.View para hacer fade-in */}
            <Animated.View style={{ flex: 1, opacity: fade }} renderToHardwareTextureAndroid={true}>
              <WebView
                key={navKey}
                ref={webRef}
                source={{
                  uri: WEBAPP_URL,
                  headers: {
                    // Manda userId en el header Cookie del PRIMER request
                    Cookie: "userId=" + encodeURIComponent(userId) + "; Path=/"
                  }
                }}
                // fondo sólido para evitar destellos al pintar
                style={{ backgroundColor: "#fff" }}

                // Quitar indicador nativo de scroll, así lo manejas desde la pag web si lo quieres
                showsVerticalScrollIndicator={false}      // iOS y, en varias versiones, Android
                showsHorizontalScrollIndicator={false}    // iOS y, en varias versiones, Android
                overScrollMode="never"                    // Android: sin glow ni rebote

                // Habilitar soporte de cookies del sistema para mayor consistencia
                sharedCookiesEnabled={true}
                thirdPartyCookiesEnabled={true}

                // cache para suavizar navegaciones intra-sitio
                cacheEnabled={true}
                cacheMode="LOAD_DEFAULT"

                // evita abrir nuevas ventanas (target=_blank) que causan parpadeo
                setSupportMultipleWindows={false}

                // Además, deja la cookie en document.cookie para navegaciones siguientes
                injectedJavaScriptBeforeContentLoaded={
                  "(function(){try{var name="+JSON.stringify("userId")+";var value="+JSON.stringify(userId)+";if(value){var exp=new Date();exp.setDate(exp.getDate()+"+COOKIE_DAYS+");var c=name+'='+value+'; path=/; expires='+exp.toUTCString()+'; SameSite=Lax';if(location.protocol==='https:') c+='; Secure';document.cookie=c;} }catch(e){};true;})();"
                }
                injectedJavaScript={
                  // postMessage + reemplazo de window.open para navegar en la misma vista
                  "(function(){try{window.open=function(url){location.href=url};}catch(e){};try{window.ReactNativeWebView&&window.ReactNativeWebView.postMessage('cookie-set');}catch(e){};true;})();"
                }
                onMessage={(e)=>{ if(e?.nativeEvent?.data==='cookie-set' && !didInitReload){ handleCookieAck(); } }}

                // Intercepta cuando el usuario navega a LOGOUT_URL
                onShouldStartLoadWithRequest = {
                  (req) => {
                      // detecta logout de forma robusta
                      const raw = (req?.mainDocumentURL || req?.url || "").toLowerCase();
                      const url = raw.split("?")[0].split("#")[0];
                      if (url.startsWith(LOGOUT_URL.toLowerCase()) || url.includes("/app-logout.php")) {
                          clientSideLogout(); // borra cookies + limpia estado + banner
                          return false; // cancela la carga de esa página
                      }
                      return true;
                  }
                }

                onError={({ nativeEvent }) => {
                    console.warn("WebView onError:", nativeEvent);
                    const d = nativeEvent?.description || "";
                    // por si no hay internet
                    if (d.includes("ERR_INTERNET_DISCONNECTED")) {
                        setErrorMsg("No internet connection");
                        setHasNetError(true);
                    } else {
                        setErrorMsg(d || "Error loading page");
                    }
                }}
                onHttpError={({ nativeEvent }) => {
                    console.warn("WebView onHttpError:", nativeEvent); // Log para debug
                    const msg = (nativeEvent?.statusCode ? "Error " + nativeEvent.statusCode + ": " : "") + (nativeEvent?.description || "HTTP error");
                    setErrorMsg(msg);
                    if (nativeEvent?.statusCode === 0) setHasNetError(true);
                }}

                 // forzar render por GPU en Android (reduce parpadeos en muchos devices)
                androidLayerType="hardware"
                // desactiva pull-to-refresh nativo que puede provocar flashes (Android)
                pullToRefreshEnabled={false}

                // antes de onLoadEnd
                onLoadStart={() => {
                    setWvLoading(true);    // muestra overlay
                    fade.setValue(0);      // reinicia opacidad
                }}
                // al terminar la carga, ejecuta el fade-in y quita overlay
                onLoadEnd={runFadeIn}
                //  opcional si hay flicker en algunos Android:
                // androidLayerType="software"
              />
            </Animated.View>

            {errorMsg ? (
              <View style={styles.errToast}>
                  <Text style={styles.errToastText}>{errorMsg}</Text>
                  <TouchableOpacity onPress={() => setErrorMsg("")} style={styles.errToastCloseBtn}>
                      <Text style={styles.errToastClose}>×</Text>
                  </TouchableOpacity>
              </View>
            ) : null}

          </View>
        );
      }


    // el árbol de login/registro
    return (
      <SafeAreaView style={styles.safe}>
        <KeyboardAvoidingView
          style={{ flex: 1 }}
          behavior={Platform.OS === "ios" ? "padding" : undefined}
        >
          <View style={styles.container}>
        
            <Image source={require("./logo.png")} style={styles.logo} resizeMode="contain" />

            <Text style={styles.title}>{mode === "login" ? "Welcome Back!" : "Create Account"}</Text>

            <TextInput
              style={styles.input}
              placeholder="Email"
              placeholderTextColor="#9AA0A6" //color fijo sin depender de theme claro oscuro del telefono
              autoCapitalize="none"
              keyboardType="email-address"
              value={email}
              onChangeText={setEmail}
            />

            {mode === "register" && (
              <TextInput
                style={styles.input}
                placeholder="Name"
                placeholderTextColor="#9AA0A6" //color fijo sin depender de theme claro oscuro del telefono
                value={name}
                onChangeText={setName}
              />
            )}

            <View style={styles.inputPwdWrap}>
               <TextInput
                 style={[styles.input, styles.inputPwd]}       // paddingRight extra
                 placeholder="Password"
                 placeholderTextColor="#9AA0A6" //color fijo sin depender de theme claro oscuro del telefono
                 secureTextEntry={!showPwd}                    // alterna visibilidad
                 autoCapitalize="none"
                 value={password}
                 onChangeText={setPassword}
                />
                <TouchableOpacity
                 style={styles.eyeBtn}
                 onPress={() => setShowPwd(!showPwd)}          // alternar
                 accessibilityLabel={showPwd ? "Hide" : "Show"}
                >
                {/* si tienes iconos, descomenta la línea de Image y borra el Text */}
                {/* <Image source={showPwd ? require("./eye-off.png") : require("./eye.png")} style={styles.eyeIcon} /> */}         
                <Text style={styles.eyeTxt}>{showPwd ? "Hide" : "Show"}</Text>
                </TouchableOpacity>
            </View>

            <TouchableOpacity style={styles.btn} onPress={doAuth} disabled={loading}>
              {loading ? (
                <ActivityIndicator color="#fff" />
              ) : (
                <Text style={styles.btnText}>{mode === "login" ? "Sign In" : "Sign Up"}</Text>
              )}
            </TouchableOpacity>

            <TouchableOpacity onPress={() => setMode(mode === "login" ? "register" : "login")}>
              <Text style={styles.link}>
                {mode === "login"
                  ? "Don't have an account? Sign up"
                  : "Already have an account? Sign in"}
              </Text>
            </TouchableOpacity>

            {errorMsg ? (
                <View style={styles.errToast}>
                    <Text style={styles.errToastText}>{errorMsg}</Text>
                    <TouchableOpacity onPress={() => setErrorMsg("")} style={styles.errToastCloseBtn}>
                        <Text style={styles.errToastClose}>×</Text>
                    </TouchableOpacity>
                </View>
            ) : null}

          </View>
        </KeyboardAvoidingView>
      </SafeAreaView>
    );
}
registerRootComponent(App);

// =====================
// Pseudo CSS
// =====================
const styles = StyleSheet.create({
    
    safe: {
        flex: 1,
        backgroundColor: "#eef1f3"
    },
    container: {
        flex: 1,
        justifyContent: "center",
        padding: 20
    },
    //pag login
    logo: {
        alignSelf: "center",
        width: 200,
        height: 200,
        marginBottom: 40
    },
    title: {
        fontSize: 30,
        fontWeight: "bold",
        textAlign: "center",
        marginBottom: 16
    },
    input: {
        borderWidth: 1,
        borderColor: "#ccc",
        borderRadius: 10,
        padding: 12,
        marginBottom: 12,
        backgroundColor: "#fff",
    },
    btn: {
        backgroundColor: "#111",
        paddingVertical: 14,
        paddingHorizontal: 20,
        borderRadius: 10,
        alignItems: "center",
        marginTop: 4,
    },
    btnText: {
        fontSize: 20,
        color: "#fff",
        fontWeight: "bold"
    },
    link: {
        fontSize: 20,
        color: "#1f6feb",
        marginTop: 16,
        textAlign: "center"
    },
    // popup de error
    errToast: {
        position: "absolute",
        left: 12,
        right: 12,
        top: 100,
        zIndex: 10,
        elevation: 10,
        backgroundColor: "#F44336",// rojo agradable
        borderRadius: 10,
        paddingVertical: 10,
        paddingHorizontal: 12,
        marginBottom: 12,
        flexDirection: "row",
        alignItems: "center",
        justifyContent: "space-between"
    },
    errToastText: {
        color: "#fff",
        fontWeight: "bold",
        flexShrink: 1
    },
    errToastCloseBtn: {
        marginLeft: 10,
        paddingHorizontal: 8,
        paddingVertical: 2
    },
    errToastClose: {
        color: "#fff",
        fontSize: 18,
        fontWeight: "bold"
    },
    // Input de password para ubicar el "ojo"
    inputPwdWrap: {
        position: "relative"
    },
    // Espacio a la derecha para no tapar el texto con el botón
    inputPwd: {
        paddingRight: 60
    },
    // botón "ojo" alineado a la derecha, centrado verticalmente
    eyeBtn: {
        position: "absolute",
        right: 12,
        top: 13,
        justifyContent: "center"
    },
    // estilo del texto del botón (si no usas icono)
    eyeTxt: {
        color: "#1f6feb",
        fontWeight: "bold"
    },
    // Estilo del icono (si usas eye.png / eye-off.png)
    eyeIcon: {
        width: 22,
        height: 22
    },
    // Overlay de carga de pantalla completa con fondo sólido
    loadingOverlay: {
        ...StyleSheet.absoluteFillObject,
        backgroundColor: "#eef1f3", // mismo color que tu sitio para no notar el cambio
        alignItems: "center",
        justifyContent: "center",
        zIndex: 20,
        elevation: 20
    },
    //Pag de error sin conexion
    offlineWrap: {
        flex: 1,
        alignItems: "center",
        justifyContent: "center",
        padding: 24,
        gap: 12,
    },
    offlineTitle: {
        fontSize: 24,
        fontWeight: "bold",
        marginBottom: 4,
    },
    offlineText: {
        fontSize: 16,
        textAlign: "center",
        marginBottom: 12,
        opacity: 0.8,
    },
    btnOutline: {
        backgroundColor: "transparent",
        borderWidth: 1,
        borderColor: "#111",
    },
    btnOutlineText: {
        color: "#111",
    },
    
});